<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Calendar Notes</title>

<style>
*{box-sizing:border-box;}
body { font-family: Arial, sans-serif; margin: 10px; }

.container { display: flex; gap: 10px; flex-wrap: wrap; }

/* LEFT PANEL */
.left-panel {
  width: 30%;
  min-width: 250px;
  border: 1px solid #ccc;
  padding: 10px;
  height: 530px;
  overflow-y: auto;
  background: #f9f9f9;
  border-radius: 6px;
}
.right-panel { width: 68%; min-width: 300px; }

.left-panel h3 {
  margin-top: 0;
  font-size: 16px;
  text-align:center;
  background:#007bff;
  color:white;
  padding:8px;
  border-radius:4px;
}

.note-date {
  margin: 8px 0 3px;
  font-weight:bold;
  background:#eee;
  padding:5px;
  border-radius:4px;
}
.note-item {
  padding-left: 10px;
  font-size: 14px;
  border-left: 2px solid #007bff;
  margin-bottom: 3px;
}

.controls { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; flex-wrap: wrap; }
.controls button {
  padding:8px 12px;font-size:14px;margin:5px 0;border:none;background:#007bff;color:white;border-radius:4px;
}
.controls button:hover { background:#0069d9; }

#header { font-size:18px;font-weight:bold;margin:5px; }

table { border-collapse: collapse; width: 100%; min-width: 350px; }
th, td { border:1px solid #999;padding:10px;text-align:center;font-size:14px; position:relative; }
td:hover{ background:#eee;cursor:pointer; }

td.has-note{ background:#ffe4a1!important;font-weight:bold; }

/* Today Highlight */
td.today {
  background:#4caf50 !important;
  color:#fff !important;
  font-weight:bold;
  border:2px solid #2e7d32;
}

/* Badge */
.badge {
  display:inline-block;
  font-size:11px;
  padding:2px 6px;
  background:#007bff;
  color:#fff;
  border-radius:50%;
  margin-left:4px;
}

#noteModal {
  display:none;background:#fff;padding:20px;border:1px solid #aaa;
  position:fixed;top:20%;left:50%;transform:translateX(-50%);
  box-shadow:0 0 15px rgba(0,0,0,0.3);z-index:10;width:90%;max-width:320px;border-radius:8px;
}
#overlay { display:none;position:fixed;width:100vw;height:100vh;background:rgba(0,0,0,0.35);top:0;left:0;z-index:5; }

input#newNote { width:100%;padding:8px;margin:10px 0;font-size:14px;border:1px solid #aaa;border-radius:4px; }
button.addBtn,button.closeBtn { width:100%;padding:8px;border:none;border-radius:4px;color:white;margin-top:5px;cursor:pointer; }
button.addBtn{ background:#28a745; } button.addBtn:hover{ background:#218838; }
button.closeBtn{ background:#dc3545; } button.closeBtn:hover{ background:#c82333; }

@media(max-width:800px){
 .left-panel,.right-panel{ width:100%; }
}
</style>
</head>

<body>
<div class="container">

  <!-- LEFT PANEL -->
  <div class="left-panel">
    <h3>Notes List</h3>

    <button onclick="toggleCalendar()" style="width:100%;padding:8px;background:#007bff;color:white;border:none;border-radius:4px;margin-bottom:10px" id="btnCalendar">
      View Calendar
    </button>

    <div style="margin-bottom:10px;">
      <select id="searchYear" style="width:32%;padding:5px;">
        <option value="">Year</option>
      </select>

      <select id="searchMonth" style="width:32%;padding:5px;">
        <option value="">Month</option>
        <option value="01">Jan</option><option value="02">Feb</option>
        <option value="03">Mar</option><option value="04">Apr</option>
        <option value="05">May</option><option value="06">Jun</option>
        <option value="07">Jul</option><option value="08">Aug</option>
        <option value="09">Sep</option><option value="10">Oct</option>
        <option value="11">Nov</option><option value="12">Dec</option>
      </select>

      <input type="number" id="searchDate" placeholder="Day" min="1" max="31" style="width:32%;padding:5px;">

      <button onclick="filterNotes()" style="width:100%;padding:6px;background:#28a745;color:white;border:none;border-radius:4px;margin-top:5px;">
        Search
      </button>

      <button onclick="resetNotes()" style="width:100%;padding:6px;background:#6c757d;color:white;border:none;border-radius:4px;margin-top:5px;">
        Reset
      </button>
    </div>

    <div id="noteListAll">Loading...</div>
  </div>

  <!-- CALENDAR -->
  <div id="calendarSection" style="display:none;">
    <div class="right-panel">
      <div class="controls">
        <button onclick="changeMonth(-1)">Prev</button>
        <span id="header"></span>
        <button onclick="changeMonth(1)">Next</button>
        <button onclick="goToday()" style="background:#28a745;">Today</button>
      </div>
      <table id="calendar"></table>
    </div>
  </div>
</div>

<!-- MODAL -->
<div id="overlay" onclick="closeModal()"></div>
<div id="noteModal">
  <h3>Notes: <span id="selDate"></span></h3>
  <ul id="noteList"></ul>
  <input type="text" id="newNote" placeholder="Type note...">
  <button class="addBtn" onclick="saveNote()">Add Note</button>
  <button class="closeBtn" onclick="closeModal()">Close</button>
</div>

<script>
let today=new Date();
let currentYear=today.getFullYear();
let currentMonth=today.getMonth();
let selectedDate="";

function getMonthName(m){
 return ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"][m];
}

async function drawCalendar(y,m){
 document.getElementById("header").textContent=getMonthName(m)+" "+y;
 const cal=document.getElementById("calendar");
 cal.innerHTML="";

 let res=await fetch(`check_notes.php?year=${y}&month=${String(m+1).padStart(2,"0")}`);
 let notes=await res.json();

 const days=["Mon","Tue","Wed","Thu","Fri","Sat","Sun"];
 let html=`<thead><tr>${days.map(d=>`<th>${d}</th>`).join("")}</tr></thead><tbody><tr>`;

 let first=new Date(y,m,1);
 let start=(first.getDay()+6)%7;

 for(let i=0;i<start;i++) html+="<td></td>";

 let last=new Date(y,m+1,0);

 for(let d=1;d<=last.getDate();d++){
   let dateKey=`${y}-${String(m+1).padStart(2,"0")}-${String(d).padStart(2,"0")}`;
   let count = notes[dateKey] ?? 0;
   let isToday = y === today.getFullYear() && m === today.getMonth() && d === today.getDate();

   let cls = "";
   if(count>0) cls="has-note";
   if(isToday) cls += " today";

   let badge = count>0 ? `<span class='badge'>${count}</span>` : "";

   if((start+d-1)%7===0 && d>1) html+="</tr><tr>";

   html += `<td class="${cls}" onclick="openNotes('${dateKey}')">${d} ${badge}</td>`;
 }

 cal.innerHTML=html+"</tr></tbody>";
}

(function(){
 let yr=document.getElementById("searchYear");
 let tYear=new Date().getFullYear();
 for(let i=tYear;i>=tYear-5;i--){
   let opt=document.createElement("option");
   opt.value=i; opt.textContent=i; yr.appendChild(opt);
 }
})();

function renderNotes(rows){
 let html="", lastDate="";
 rows.forEach(n=>{
   if(lastDate!==n.note_date){
     lastDate=n.note_date;
     html+=`<div class="note-date">${n.note_date}</div>`;
   }
   html+=`<div class="note-item">• ${n.note_text}</div>`;
 });
 document.getElementById("noteListAll").innerHTML=html || "No notes found";
}

async function loadAllNotes(){
 let res=await fetch("get_all_notes.php");
 let data=await res.json();
 renderNotes(data);
}

async function filterNotes(){
 let y=document.getElementById("searchYear").value;
 let m=document.getElementById("searchMonth").value;
 let d=document.getElementById("searchDate").value;

 let res=await fetch("get_all_notes.php");
 let data=await res.json();

 let filtered=data.filter(n=>{
   if(y && !n.note_date.startsWith(y)) return false;
   if(m && n.note_date.substr(5,2)!=m) return false;
   if(d && n.note_date.substr(8,2)!=String(d).padStart(2,"0")) return false;
   return true;
 });

 renderNotes(filtered);
}

function resetNotes(){
 document.getElementById("searchYear").value="";
 document.getElementById("searchMonth").value="";
 document.getElementById("searchDate").value="";
 loadAllNotes();
}

function changeMonth(x){
 currentMonth+=x;
 if(currentMonth>11){currentMonth=0;currentYear++;}
 if(currentMonth<0){currentMonth=11;currentYear--;}
 drawCalendar(currentYear,currentMonth);
}

function goToday(){
 currentYear=today.getFullYear();
 currentMonth=today.getMonth();
 drawCalendar(currentYear,currentMonth);
}

async function openNotes(date){
 selectedDate=date;
 document.getElementById("selDate").textContent=date;

 let res=await fetch(`get_notes.php?date=${date}`);
 let notes=await res.json();

 document.getElementById("noteList").innerHTML =
   notes.map(n=>`<li>${n.note_text}</li>`).join("");

 document.getElementById("overlay").style.display="block";
 document.getElementById("noteModal").style.display="block";
}

function closeModal(){
 document.getElementById("noteModal").style.display="none";
 document.getElementById("overlay").style.display="none";
}

async function saveNote(){
 let text=document.getElementById("newNote").value;
 if(!text.trim()) return;

 await fetch("save_note.php",{
   method:"POST",
   headers:{ "Content-Type":"application/x-www-form-urlencoded" },
   body:`date=${selectedDate}&note=${encodeURIComponent(text)}`
 });

 document.getElementById("newNote").value="";
 closeModal();
 loadAllNotes();
 drawCalendar(currentYear,currentMonth);
 openNotes(selectedDate);
}

function toggleCalendar(){
 let cal=document.getElementById("calendarSection");
 let btn=document.getElementById("btnCalendar");
 if(cal.style.display==="none"){
   cal.style.display="block";
   btn.textContent="Hide Calendar";
   drawCalendar(currentYear,currentMonth);
 } else {
   cal.style.display="none";
   btn.textContent="View Calendar";
 }
}

loadAllNotes();
</script>
</body>
</html>

