<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Calendar Notes</title>

<style>
*{box-sizing:border-box;}
html { height: 100%; }
body { 
  font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
  margin: 10px; 
  background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
  background-attachment: fixed;
  color: #334155; 
  min-height: 100%;
}

.container { display: flex; gap: 10px; flex-wrap: wrap; }

/* LEFT PANEL */
.left-panel {
  width: 30%;
  min-width: 250px;
  border: 1px solid #ffffff;
  padding: 10px;
  height: 530px;
  overflow-y: auto;
  background: rgba(255, 255, 255, 0.9);
  backdrop-filter: blur(10px);
  border-radius: 8px;
  box-shadow: 0 4px 6px rgba(0,0,0,0.05), 0 1px 3px rgba(0,0,0,0.1);
}
.right-panel { width: 68%; min-width: 300px; }

.left-panel h3 {
  margin-top: 0;
  font-size: 16px;
  text-align:center;
  background: linear-gradient(to right, #3b82f6, #2563eb);
  color:white;
  padding:10px;
  border-radius:6px;
  box-shadow: 0 2px 4px rgba(59, 130, 246, 0.3);
}

.note-date {
  margin: 8px 0 3px;
  font-weight:bold;
  background: #f1f5f9;
  color: #3b82f6;
  padding:6px;
  border-radius:4px;
}
.note-item {
  padding-left: 10px;
  font-size: 14px;
  border-left: 3px solid #3b82f6;
  margin-bottom: 4px;
  color: #475569;
  background: rgba(255,255,255,0.5);
  padding: 4px 4px 4px 10px;
  border-radius: 0 4px 4px 0;
}

.controls { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; flex-wrap: wrap; }
.controls button {
  padding:8px 14px;font-size:14px;margin:5px 0;border:none;
  background: linear-gradient(to bottom, #3b82f6, #2563eb);
  color:white;border-radius:6px; cursor: pointer;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
  transition: transform 0.1s, box-shadow 0.1s;
}
.controls button:hover { 
  background: linear-gradient(to bottom, #2563eb, #1d4ed8); 
  transform: translateY(-1px);
  box-shadow: 0 4px 6px rgba(0,0,0,0.15);
}
.controls button:active { transform: translateY(0); }

#header { font-size:20px;font-weight:700;margin:5px; color: #1e293b; text-shadow: 0 1px 0 rgba(255,255,255,0.5); }

table { 
  border-collapse: separate; 
  border-spacing: 0;
  width: 100%; 
  min-width: 350px; 
  background: rgba(255, 255, 255, 0.95); 
  border-radius: 8px; 
  overflow: hidden; 
  box-shadow: 0 4px 6px rgba(0,0,0,0.05), 0 1px 3px rgba(0,0,0,0.1); 
}
th, td { border:1px solid #e2e8f0;padding:10px;text-align:center;font-size:14px; position:relative; }
th { background: #f8fafc; color: #64748b; font-weight: 600; text-transform: uppercase; font-size: 12px; letter-spacing: 0.5px; }
td:hover{ background:#f1f5f9;cursor:pointer; }

td.has-note{ background:#eff6ff !important; font-weight:bold; color: #1d4ed8; }

/* Today Highlight */
td.today {
  background:#ecfdf5 !important;
  color:#047857 !important;
  font-weight:bold;
  border:2px solid #10b981;
}

/* Badge */
.badge {
  display:inline-block;
  font-size:10px;
  padding:2px 6px;
  background:#ef4444;
  color:#fff;
  border-radius:10px;
  margin-left:4px;
  box-shadow: 0 2px 4px rgba(239, 68, 68, 0.3);
}

#noteModal {
  display:none;background:#fff;padding:25px;border:none;
  position:fixed;top:50%;left:50%;transform:translate(-50%, -50%);
  box-shadow:0 20px 25px -5px rgba(0,0,0,0.1), 0 10px 10px -5px rgba(0,0,0,0.04);
  z-index:10;width:90%;max-width:350px;border-radius:12px;
}
#overlay { display:none;position:fixed;width:100vw;height:100vh;background:rgba(15, 23, 42, 0.4);backdrop-filter: blur(2px);top:0;left:0;z-index:5; }

input#newNote { width:100%;padding:10px;margin:15px 0;font-size:14px;border:1px solid #cbd5e1;border-radius:6px; background: #f8fafc; }
input#newNote:focus { outline:none; border-color: #3b82f6; background: #fff; }

button.addBtn,button.closeBtn { width:100%;padding:10px;border:none;border-radius:6px;color:white;margin-top:5px;cursor:pointer; font-weight: 600; }
button.addBtn{ background: linear-gradient(to bottom, #10b981, #059669); } 
button.addBtn:hover{ background: linear-gradient(to bottom, #059669, #047857); }
button.closeBtn{ background: transparent; color: #64748b; border: 1px solid #cbd5e1; } 
button.closeBtn:hover{ background: #f1f5f9; color: #334155; }

/* MOBILE RESPONSIVE TWEAKS */
@media(max-width:800px){
  .container {
    flex-direction: column; /* Stack vertically */
  }
  
  /* Order: 1 appears first, 2 appears second */
  #calendarSection {
    order: 1; /* Calendar FIRST */
    /* display: block;  <-- REMOVED forced block, so toggle works */
    margin-bottom: 20px;
  }
  
  .left-panel {
    order: 2; /* Notes List SECOND */
    width: 100%;
    height: auto; /* Allow auto height on mobile */
    max-height: 500px;
  }
  
  .right-panel {
    width: 100%;
  }
  
  /* Show the "View Calendar" button on mobile */
  #btnCalendar {
    display: block; /* Restored button */
  }
}
</style>
</head>

<body>
<div class="container">

  <!-- LEFT PANEL (Notes List) -->
  <div class="left-panel">
    <h3>Notes List</h3>

    <button onclick="toggleCalendar()" style="width:100%;padding:10px;margin-bottom:15px" id="btnCalendar">
      View Calendar
    </button>

    <div style="margin-bottom:15px;">
      <select id="searchYear" style="width:32%;padding:8px;border:1px solid #cbd5e1;border-radius:6px;">
        <option value="">Year</option>
      </select>

      <select id="searchMonth" style="width:32%;padding:8px;border:1px solid #cbd5e1;border-radius:6px;">
        <option value="">Month</option>
        <option value="01">Jan</option><option value="02">Feb</option>
        <option value="03">Mar</option><option value="04">Apr</option>
        <option value="05">May</option><option value="06">Jun</option>
        <option value="07">Jul</option><option value="08">Aug</option>
        <option value="09">Sep</option><option value="10">Oct</option>
        <option value="11">Nov</option><option value="12">Dec</option>
      </select>

      <input type="number" id="searchDate" placeholder="Day" min="1" max="31" style="width:32%;padding:8px;border:1px solid #cbd5e1;border-radius:6px;">

      <button onclick="filterNotes()" style="width:100%;padding:8px;background: linear-gradient(to bottom, #10b981, #059669);margin-top:8px;">
        Search
      </button>

      <button onclick="resetNotes()" style="width:100%;padding:8px;background: linear-gradient(to bottom, #64748b, #475569);margin-top:5px;">
        Reset
      </button>
    </div>

    <div id="noteListAll">Loading...</div>
  </div>

  <!-- CALENDAR (Right Panel) -->
  <div id="calendarSection" style="display:none;"> <!-- Initially hidden on desktop and mobile -->
    <div class="right-panel">
      <div class="controls">
        <button onclick="changeMonth(-1)">Prev</button>
        <span id="header"></span>
        <button onclick="changeMonth(1)">Next</button>
        <button onclick="goToday()" style="background: linear-gradient(to bottom, #10b981, #059669);">Today</button>
      </div>
      <table id="calendar"></table>
    </div>
  </div>
</div>

<!-- MODAL -->
<div id="overlay" onclick="closeModal()"></div>
<div id="noteModal">
  <h3>Notes: <span id="selDate" style="color:#3b82f6"></span></h3>
  <ul id="noteList"></ul>
  <input type="text" id="newNote" placeholder="Type note...">
  <button class="addBtn" onclick="saveNote()">Add Note</button>
  <button class="closeBtn" onclick="closeModal()">Close</button>
</div>

<script>
let today=new Date();
let currentYear=today.getFullYear();
let currentMonth=today.getMonth();
let selectedDate="";

function getMonthName(m){
 return ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"][m];
}

async function drawCalendar(y,m){
 document.getElementById("header").textContent=getMonthName(m)+" "+y;
 const cal=document.getElementById("calendar");
 cal.innerHTML="";

 let res=await fetch(`check_notes.php?year=${y}&month=${String(m+1).padStart(2,"0")}`);
 let notes=await res.json();

 const days=["Mon","Tue","Wed","Thu","Fri","Sat","Sun"];
 let html=`<thead><tr>${days.map(d=>`<th>${d}</th>`).join("")}</tr></thead><tbody><tr>`;

 let first=new Date(y,m,1);
 let start=(first.getDay()+6)%7;

 for(let i=0;i<start;i++) html+="<td></td>";

 let last=new Date(y,m+1,0);

 for(let d=1;d<=last.getDate();d++){
   let dateKey=`${y}-${String(m+1).padStart(2,"0")}-${String(d).padStart(2,"0")}`;
   let count = notes[dateKey] ?? 0;
   let isToday = y === today.getFullYear() && m === today.getMonth() && d === today.getDate();

   let cls = "";
   if(count>0) cls="has-note";
   if(isToday) cls += " today";

   let badge = count>0 ? `<span class='badge'>${count}</span>` : "";

   if((start+d-1)%7===0 && d>1) html+="</tr><tr>";

   html += `<td class="${cls}" onclick="openNotes('${dateKey}')">${d} ${badge}</td>`;
 }

 cal.innerHTML=html+"</tr></tbody>";
}

(function(){
 let yr=document.getElementById("searchYear");
 let tYear=new Date().getFullYear();
 for(let i=tYear;i>=tYear-5;i--){
   let opt=document.createElement("option");
   opt.value=i; opt.textContent=i; yr.appendChild(opt);
 }
})();

function renderNotes(rows){
 let html="", lastDate="";
 rows.forEach(n=>{
   if(lastDate!==n.note_date){
     lastDate=n.note_date;
     html+=`<div class="note-date">${n.note_date}</div>`;
   }
   html+=`<div class="note-item">• ${n.note_text}</div>`;
 });
 document.getElementById("noteListAll").innerHTML=html || "No notes found";
}

async function loadAllNotes(){
 let res=await fetch("get_all_notes.php");
 let data=await res.json();
 renderNotes(data);
}

async function filterNotes(){
 let y=document.getElementById("searchYear").value;
 let m=document.getElementById("searchMonth").value;
 let d=document.getElementById("searchDate").value;

 let res=await fetch("get_all_notes.php");
 let data=await res.json();

 let filtered=data.filter(n=>{
   if(y && !n.note_date.startsWith(y)) return false;
   if(m && n.note_date.substr(5,2)!=m) return false;
   if(d && n.note_date.substr(8,2)!=String(d).padStart(2,"0")) return false;
   return true;
 });

 renderNotes(filtered);
}

function resetNotes(){
 document.getElementById("searchYear").value="";
 document.getElementById("searchMonth").value="";
 document.getElementById("searchDate").value="";
 loadAllNotes();
}

function changeMonth(x){
 currentMonth+=x;
 if(currentMonth>11){currentMonth=0;currentYear++;}
 if(currentMonth<0){currentMonth=11;currentYear--;}
 drawCalendar(currentYear,currentMonth);
}

function goToday(){
 currentYear=today.getFullYear();
 currentMonth=today.getMonth();
 drawCalendar(currentYear,currentMonth);
}

async function openNotes(date){
 selectedDate=date;
 document.getElementById("selDate").textContent=date;

 let res=await fetch(`get_notes.php?date=${date}`);
 let notes=await res.json();

 document.getElementById("noteList").innerHTML =
   notes.map(n=>`<li>${n.note_text}</li>`).join("");

 document.getElementById("overlay").style.display="block";
 document.getElementById("noteModal").style.display="block";
}

function closeModal(){
 document.getElementById("noteModal").style.display="none";
 document.getElementById("overlay").style.display="none";
}

async function saveNote(){
 let text=document.getElementById("newNote").value;
 if(!text.trim()) return;

 await fetch("save_note.php",{
   method:"POST",
   headers:{ "Content-Type":"application/x-www-form-urlencoded" },
   body:`date=${selectedDate}&note=${encodeURIComponent(text)}`
 });

 document.getElementById("newNote").value="";
 closeModal();
 loadAllNotes();
 drawCalendar(currentYear,currentMonth);
 openNotes(selectedDate);
}

function toggleCalendar(){
 let cal=document.getElementById("calendarSection");
 let btn=document.getElementById("btnCalendar");
 if(cal.style.display==="none"){
   cal.style.display="block";
   btn.textContent="Hide Calendar";
   drawCalendar(currentYear,currentMonth);
 } else {
   cal.style.display="none";
   btn.textContent="View Calendar";
 }
}

// Removed the auto-show logic for mobile
// if(window.innerWidth <= 800) { ... }

loadAllNotes();
</script>
</body>
</html>
